window.onload = function() {
    // Initialisation de la carte centrée sur Ouagadougou avec un zoom de 10
    var map = L.map('map').setView([12.3711, -1.5197], 10);
    var destinationMarker, sourceMarker;

    // Chargement des tuiles OpenStreetMap
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors',
        minZoom: 1,
        maxZoom: 20
    }).addTo(map);

    // Initialisation du contrôle de routage avec OSRM
    var control = L.Routing.control({
        lineOptions: {
            styles: [{ color: '#839c49', opacity: 1, weight: 7 }]
        },
        router: new L.Routing.osrmv1({
            language: 'fr',
            profile: 'car'
        }),
        waypoints: [],
        createMarker: function(i, waypoint, n) {
            var marker = L.marker(waypoint.latLng, { draggable: true })
                .bindPopup(i === 0 ? "Point de départ" : "Point d'arrivée")
                .on('dragend', function(e) {
                    var newLatLng = e.target.getLatLng();
                    control.spliceWaypoints(i, 1, newLatLng);
                    updateMarkers(); // Mise à jour des marqueurs après déplacement
                });
            return marker;
        },
        routeWhileDragging: true,
        show: false,
        draggableWaypoints: true
    }).addTo(map);

    // Fonction pour mettre à jour les marqueurs (départ et arrivée)
    function updateMarkers() {
        if (sourceMarker) {
            sourceMarker.setLatLng(control.getWaypoints()[0].latLng);
        }
        if (destinationMarker) {
            destinationMarker.setLatLng(control.getWaypoints()[1].latLng);
        }
    }

    // Fonction pour afficher les suggestions de recherche
    function afficherSuggestions(input, containerSuggestions, inputElement) {
        if (input.length < 2) {
            containerSuggestions.innerHTML = '';
            return;
        }

        var url = `https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(input)}&countrycodes=BF&accept-language=fr&addressdetails=1&limit=50`;

        fetch(url)
            .then(response => response.json())
            .then(data => {
                containerSuggestions.innerHTML = '';
                if (data.length > 0) {
                    data.forEach(result => {
                        var suggestion = document.createElement('div');
                        suggestion.className = 'suggestion';
                        suggestion.textContent = result.display_name;
                        suggestion.onclick = function() {
                            inputElement.value = result.display_name;
                            var latLng = [result.lat, result.lon];

                            // Gestion des marqueurs pour le départ ou l'arrivée
                            if (inputElement.id === 'source-search') {
                                if (sourceMarker) map.removeLayer(sourceMarker);
                                sourceMarker = createMarker(latLng, "Point de départ", 0);
                            } else {
                                if (destinationMarker) map.removeLayer(destinationMarker);
                                destinationMarker = createMarker(latLng, "Point d'arrivée", 1);
                            }

                            containerSuggestions.innerHTML = '';
                        };
                        containerSuggestions.appendChild(suggestion);
                    });
                }
            })
            .catch(error => console.error("Erreur lors de la récupération des suggestions : ", error));
    }

    // Fonction pour créer un marqueur (utilisé pour le départ et l'arrivée)
    function createMarker(latLng, popupText, waypointIndex) {
        return L.marker(latLng, { draggable: true })
            .addTo(map)
            .bindPopup(popupText)
            .openPopup()
            .on('dragend', function(e) {
                var newLatLng = e.target.getLatLng();
                control.spliceWaypoints(waypointIndex, 1, newLatLng);
                updateMarkers();
            });
    }

    // Ajout d'écouteurs pour les champs de recherche (départ et arrivée)
    document.getElementById('source-search').addEventListener('input', function() {
        afficherSuggestions(this.value, document.getElementById('source-suggestions'), this);
    });

    document.getElementById('destination-search').addEventListener('input', function() {
        afficherSuggestions(this.value, document.getElementById('destination-suggestions'), this);
    });

    // Fonction pour définir les points de départ et d'arrivée
    document.getElementById('set-destinations-btn').onclick = function() {
        var sourceQuery = document.getElementById('source-search').value;
        var destinationQuery = document.getElementById('destination-search').value;

        if (sourceQuery && destinationQuery) {
            setDestination(sourceQuery, destinationQuery);
        } else {
            alert('Veuillez entrer une destination de départ et une destination d\'arrivée.');
        }
    };

    // Fonction pour gérer les appels à Nominatim pour définir les destinations
    function setDestination(sourceQuery, destinationQuery) {
        fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(sourceQuery)}&countrycodes=BF`)
            .then(response => response.json())
            .then(data => {
                if (data.length > 0) {
                    var sourceLatLng = [data[0].lat, data[0].lon];
                    fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(destinationQuery)}&countrycodes=BF`)
                        .then(response => response.json())
                        .then(data => {
                            if (data.length > 0) {
                                var destinationLatLng = [data[0].lat, data[0].lon];
                                control.setWaypoints([sourceLatLng, destinationLatLng]);
                                
                                sourceMarker = createMarker(sourceLatLng, "Point de départ", 0);
                                destinationMarker = createMarker(destinationLatLng, "Point d'arrivée", 1);
                                map.setView(sourceLatLng, 13);
                                afficherInstructionsItineraire(control);
                            } else {
                                alert("Destination d'arrivée non trouvée.");
                            }
                        });
                } else {
                    alert("Destination de départ non trouvée.");
                }
            });
    }

    // Fonction pour afficher les instructions d'itinéraire
    function afficherInstructionsItineraire(routingControl) {
        routingControl.on('routesfound', function(e) {
            var routes = e.routes;
            var instructionsList = document.getElementById('instructions-list');
            instructionsList.innerHTML = '';

            var distanceTotale = routes[0].summary.totalDistance / 1000;
            var tempsTotal = Math.ceil(routes[0].summary.totalTime / 60);

            instructionsList.innerHTML += `<p>Distance totale : ${distanceTotale.toFixed(2)} km</p>`;
            instructionsList.innerHTML += `<p>Temps total : ${tempsTotal} min</p>`;

            routes[0].instructions.forEach(function(instruction) {
                var distance = instruction.distance.toFixed(0);
                var instructionTemps = Math.ceil(instruction.time / 60);
                instructionsList.innerHTML += `<li>${instruction.text} (${distance}m, ${instructionTemps} min)</li>`;
            });

            document.getElementById('instructions-container').style.display = 'block';
        });
    }
};
